<?php

/**
 * Menu callback; present an administrative trackback listing.
 */
function trackback_admin_overview(&$form_state, $type = 'published') {
  // Disable Approval Queue if moderation isn't enabled
  if (!variable_get('trackback_moderation', TRUE) && $type != 'published') {
    $form = array();
    $form['trackback_message'] = array(
      '#value' => t('Approval queue is disabled, enable at : !url', array('!url' => l(t('Trackback settings'), 'admin/content/trackback/settings'))),
    );
    return $form;
  }
  if (!trackback_spam_filtering() && $type == 'spam') {
    $form = array();
    $form['trackback_message'] = array(
      '#value' => t("Trackback can't find any anti-spam module. Install one and enable at : !url", array('!url' => l(t('Trackback settings'), 'admin/content/trackback/settings'))),
    );
    return $form;
  }
  
  $operations = array();
  if ($type == 'published') {
    $operations['unpublish'] = t('Unpublish the selected trackbacks');
  }
  else {
    $operations['publish'] = t('Publish the selected trackbacks');
  }
  if ($type == 'spam') {
    $operations['trackback_as_notspam'] = t('Mark the selected trackbacks as not spam');
  }
  else {
    $operations['mark_as_spam'] = t('Mark the selected trackbacks as spam');
  }

  $operations['delete'] = t('Delete the selected trackbacks (no confirmation)');

  $form = array();
  $form['update'] = array(
    '#type' => 'fieldset',
    '#title' => t('Update options'),
    'operations' => array('#type' => 'value', '#value' => $operations)
  );
  $form['update']['operation'] = array(
    '#prefix' => '<div class="container-inline">',
    '#type' => 'select',
    '#options' => $operations
  );
  $form['update']['op'] = array(
    '#type' => 'submit',
    '#value' => t('Update'),
    '#suffix' => '</div>'
  );

  $form['trackbacks'] = array('#theme' => 'trackback_admin_table');
  $form['trackbacks']['header'] = array(
    '#type' => 'value',
    '#value' => trackback_admin_table_header($type)
  );

  switch ($type) {
    case 'published':
      $status = TRACKBACK_STATUS_PUBLISHED;
      break;    
    case 'spam':
      $status = TRACKBACK_STATUS_SPAM;
      break;
    case 'approval':
      $status[] = TRACKBACK_STATUS_UNPUBLISHED;
      $status[] = TRACKBACK_STATUS_SPAM_UNSURE;
      
  }

  $sql = 'SELECT n.nid AS nodenid, n.title AS nodetitle, tr.* FROM {trackback_received} tr INNER JOIN {node} n ON tr.nid = n.nid WHERE tr.status = %d';
  if (is_array($status)) {
    $sql .= ' OR tr.status = %d'; 
  }
  $sql .= tablesort_sql($form['trackbacks']['header']['#value']);
  drupal_set_message($sql);
  $result = pager_query($sql, 50, 0, NULL, $status);

  $form['trackbacks']['status'] = array('#tree' => TRUE);
  while ($trackback = db_fetch_object($result)) {
    $form['trackbacks']['status'][$trackback->trid] = array('#type' => 'checkbox');
    $form['trackbacks'][$trackback->trid] = array();
    $form['trackbacks'][$trackback->trid][] = array('#value' => l($trackback->nodetitle, 'node/'.$trackback->nodenid));
    $form['trackbacks'][$trackback->trid][] = array('#value' => $trackback->subject);
    $form['trackbacks'][$trackback->trid][] = array('#value' => truncate_utf8($trackback->name, 15, FALSE, TRUE));
    $form['trackbacks'][$trackback->trid][] = array('#value' => $trackback->site);
    
    if ($type == 'approval') {
      $form['trackbacks'][$trackback->trid][] = array('#value' => trackback_translate_status($trackback->status));
    }
        // $form['trackbacks'][$trackback->trid][] = array('#value' => $trackback->probability >= $spam_threshold ? t('Spam') : t('Not Spam'));

    $form['trackbacks'][$trackback->trid][] = array('#value' => format_date($trackback->created, 'small'));
    $form['trackbacks'][$trackback->trid][] = array('#value' => l(t('edit'), 'trackback/edit/'. $trackback->trid,     array('query' => drupal_get_destination())));
    $form['trackbacks'][$trackback->trid][] = array('#value' => l(t('delete'), 'trackback/delete/'. $trackback->trid, array('query' => drupal_get_destination())));
  }
  $form['pager'] = array('#value' => theme('pager', NULL, 50, 0));

  return $form;
}

/**
 * Validate function for admin_overview form
 * @param $form_id
 * @param $form_state
 */
function trackback_admin_overview_validate($form_id, $form_state) {
  $status = array();
  if (isset($form_state['values']['status'])) {
    $status = array_diff((array)$form_state['values']['status'], array(0));
  }
  if (count($status) == 0) {
    form_set_error('', t('Please select one or more trackbacks to perform the update on.'));
    drupal_goto($_GET['q']);
  }
}

function trackback_admin_overview_submit($form_id, $form_state) {
  if (isset($form_state['values']['operations'][$form_state['values']['operation']])) {
    foreach ($form_state['values']['status'] as $trid => $value) {
      if ($value) {
        _trackback_operation($form_state['values']['operation'], $trid, FALSE);
      }
    }
    cache_clear_all();
    drupal_set_message(t('The update has been performed.'));
  }
}

/**
 * Provide the admin_overview table header
 */
function trackback_admin_table_header($type) {
  $header = array();
  $header[] = theme('table_select_header_cell');
  $header[] = array('data' => t('Node'), 'field' => 'n.nid');
  $header[] = array('data' => t('Subject'), 'field' => 'tr.subject');
  $header[] = array('data' => t('Author'), 'field' => 'tr.name');
  $header[] = array('data' => t('Host'), 'field' => 'tr.site');
  if ($type == 'approval') {  
    $header[] = array('data' => t('Status'), 'field' => 'tr.status');
  }
  
  $header[] = array('data' => t('Time'), 'field' => 'created', 'sort' => 'desc');
  $header[] = array('data' => t('Operations') , 'colspan' => '2');
  return $header;
}

function theme_trackback_admin_table($form) {
  $header = $form['header']['#value'];
  $rows = array();
  foreach (element_children($form['status']) as $key) {
    $row = array(drupal_render($form['status'][$key]));
    foreach (element_children($form[$key]) as $column_key) {
      $row[] = drupal_render($form[$key][$column_key]);
    }
    $rows[] = $row;
  }
  if (count($rows) == 0) {
    $rows[] = array(array('data' => t('No trackbacks available.'), 'colspan' => count($header)));
  }
  return theme('table', $header, $rows);
}

/*
function trackback_configure() {
  $form['trackback_auto_detection'] = array(
    '#type' => 'fieldset',
    '#title' => t('Auto-detection'),    
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
    '#description' => t('Auto-detection for pings to other sites.'),
  );
  
  $form['trackback_auto_detection']['trackback_auto_detection_enabled'] = array(
    '#type' => 'radios',
    '#default_value' => variable_get('trackback_auto_detection_enabled', 0),
    '#options' => array(
      t('Disabled'),
      t('Enabled'),
      t('Run auto-detection on cron')
    ),
    '#description' => t('If auto-detection is enabled, each URL in any posted content (whether in textile, link, or plain-text form) will be checked for a trackback URL upon submission. For each URL in the body of the posted content, trackback will check to see if that URL accepts trackbacks from other sites. If a URL accepts trackbacks, trackback will ping the trackback URL found on that page if one has been posted at that URL.<br>*note: This has the potential to take a very long time depending on the amount of links you have in your posts. Using the \'Run auto-detection on cron\' option delays the most time consuming part of the process to when cron is run on the site. This speeds perfomance when editing and creating content, but delays trackbacks until cron is run.'),    
  );
  
  $form['trackback_auto_detection']['trackback_link_only'] = array(
    '#type' => 'checkbox',
    '#title' => t('Link only'),
    '#default_value' => variable_get('trackback_link_only', 0),
    '#description' => t('If checked, auto-detection will check link only.')
  );
  
  $form['trackback_retention_settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('Trackback retention settings'),
    //'#description' => t(''),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );
  
  $form['trackback_retention_settings']['trackback_moderation'] = array(
    '#type' => 'radios',
    '#title' => t('Trackback moderation'),
    '#default_value' => variable_get('trackback_moderation', 0),
    '#options' => array(t('Disabled'), t('Enabled')),
    '#description' => t('Enabling moderation forces every received trackback to be approved before it will appear on your site. The moderation queue can then be viewed on the !linked_page.', array('!linked_page' => l(t('trackback administration page'), 'admin/content/trackback/list/approval')))
  );
  
  $form['trackback_retention_settings']['trackback_reject_oneway'] = array(
    '#type' => 'radios',
    '#title' => t('Reject one-way trackbacks'),
    '#default_value' => variable_get('trackback_reject_oneway', 0),
    '#options' => array(t('Disabled'), t('Enabled')),
    '#description' => t('If enabled, trackbacks that the sender page does not refer to your site will be rejected.')
  );

  // Setting for Trackback display (trackbacks and Trackback-URL)
  $form['trackback_display'] = array(
    '#type' => 'fieldset',
    '#title' => t('Trackback display'),
    '#description' => t('Settings to configure Trackback displaing modes'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );

}
*/

function trackback_delete_form(&$form_state, $trackback) {
  $form = array();
  $form['trackback'] = array(
    '#type' => 'value',
    '#value' => $trackback->trid,
  );
  
  $form['destination'] = array(
    '#type' => 'value',
    '#value' => $_REQUEST['destination'],
  );

  return confirm_form(
    $form,
    t('Are you sure you want to delete the trackback "%title"?', array('%title' => $trackback->subject)),
    '',
    t('This action cannot be undone.'),
    t('Delete'),
    t('Cancel')
  );
}

function trackback_delete_form_submit($form, &$form_state) {
  $trid = $form_state['values']['trackback'];
  if (trackback_do('delete', $trid)) {
    drupal_set_message(t('Trackback deleted'));
  }
  //$form_state['redirect'] = array($form_state['values']['destination'], NULL, 'trackbacks');
}

function trackback_edit_form(&$form_state, $trackback) {
  $node = node_load($trackback->nid);
  $form['trid'] = array(
    '#type' => 'hidden',
    '#value' => $trackback->trid,
  );
  $form['node'] = array(
    '#type' => 'item',
    '#title' => t('Referenced Node'),
    '#value' => l($node->title, 'node/' . $trackback->nid),
  );
  $form['subject'] = array(
    '#type' => 'textfield',
    '#title' => t('Subject'),
    '#description' => t('Usually the page title of referencing url'),
    '#default_value' => $trackback->subject,
  );

  $form['excerpt'] = array(
    '#type' => 'textarea',
    '#title' => t('Excerpt'),
    '#default_value' => $trackback->excerpt,
    '#size' => 60,
    '#maxlength' => 255,
  );

  $options = trackback_translate_status(NULL);
  if (!trackback_spam_filtering()) {
    // If not SPAM filtering, remove that OPs and force to "Unpublished"
    unset($options[TRACKBACK_STATUS_SPAM]);
    unset($options[TRACKBACK_STATUS_SPAM_UNSURE]);
    if ($trackback->status == TRACKBACK_STATUS_SPAM ||
        $trackback->status == TRACKBACK_STATUS_SPAM_UNSURE) {
      $trackback->status = TRACKBACK_STATUS_UNPUBLISHED;
    }
  }
  $form['status'] = array(
    '#type' => 'radios',
    '#title' => t('Trackback Status'),
    '#default_value' => $trackback->status,
    '#options' => $options,
  );

  $form['info'] = array(
    '#type' => 'fieldset',
    '#title' => t('Sender Info'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );
  // Creation date
  $form['info']['created'] = array(
    '#type' => 'item',
    '#title' => t('Date'),
    '#description' => t('The timestamp the Trackback was received.'),
    '#value' => format_date($trackback->created, 'large'),
  );
  $form['info']['url'] = array(
    '#type' => 'item',
    '#title' => t('URL'),
    '#description' => t('This is the url that the trackback point to.'),
    '#value' => l($trackback->url, $trackback->url),
  );
  $form['info']['ip'] = array(
    '#type' => 'item',
    '#title' => t('IP'),
    '#description' => t('This is the UP the trackback was sent from.'),
    '#value' => $trackback->site,
  );

  $form['buttons']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Save'),
  );
  $form['buttons']['delete'] = array(
    '#type' => 'submit',
    '#value' => t('Delete trackback'),
  );
  return $form;
}

/**
 * Submit callback for edit_form
 * @param <type> $form
 * @param <type> $form_state
 */
function trackback_edit_form_submit($form, &$form_state) {
  $status = $form_state['values']['status'];
  $status_old = $form['status']['#default_value'];
  // drupal_set_message(__function__ . ' From:'. $status_old . ' to:' . $status);

  if ($form_state['values']['op'] == $form_state['values']['delete']) {
    // Redirect to delete_confirm form
    drupal_goto('trackback/delete/' . $form_state['values']['trid']);
  }
  elseif ($form_state['values']['op'] == $form_state['values']['submit']) {
    $trackback = trackback_load($form_state['values']['trid']);

    // Update changed fields
    $trackback->subject = $form_state['values']['subject'];
    $trackback->excerpt = $form_state['values']['excerpt'];

    if ($status != $status_old) {
      switch ($status) {
        case TRACKBACK_STATUS_PUBLISHED:
          trackback_do('publish', $trackback);
          break;
        case TRACKBACK_STATUS_UNPUBLISHED:
          trackback_do('unpublish', $trackback);
          break;
        case TRACKBACK_STATUS_SPAM:
          trackback_do('mark-spam', $trackback);
          break;
        case TRACKBACK_STATUS_SPAM_UNSURE:
          // @TODO: what we shall do here?
          _trackback_do_update($trackback);
          break;
      }
    }else {
      // No status changes, only update the Trackback
      _trackback_do_update($trackback);
    }
  }

  return;
}

/**
 * Trakcback configuration form
 */
function trackback_config_form() {
  $form = array();
  // Receiving settings
  $form['trackback_receiving_settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('Receiving Settings'),
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );

  $form['trackback_receiving_settings']['trackback_receive_xmlrpc'] = array(
    '#type' => 'checkbox',
    '#title' => t('Enable XMLRPC receiving'),
    '#description' => t('Allow receiving trackbacks using Pingback 1.0 protocol (uses XMLRPC calls)'),
    '#default_value' => variable_get('trackback_receive_xmlrpc', FALSE),
  );
  
  $form['trackback_receiving_settings']['trackback_receive_post'] = array(
    '#type' => 'checkbox',
    '#title' => t('Enable POST/GET receiving'),
    '#description' => t('Allow receiving trackbacks using Trackback protocol (using POST requests)'),
    '#default_value' => variable_get('trackback_receive_post', TRUE),
  );
  
  if (module_exists('locale')) {
    $form['trackback_receiving_settings']['trackback_merge_within_translations'] = array(
      '#type' => 'checkbox',
      '#title' => t('Merge trackback for Translations'),
      '#description' => t('If the trackback is received on a translated node, keep the trackback associated with the source node, not the translated one. Changing this option with already received trackbacks will lead to missing trackbacks...(??)'),
      '#default_value' => variable_get('trackback_merge_within_translations', TRUE),
    );
  }
  
  $form['trackback_receiving_settings']['trackback_receive_xmlrpc_getpingbacks'] = array(
    '#type' => 'checkbox',
    '#title' => '(*EXPERIMENTAL*)'. t('Enable XMLRPC trackbacks sharing'),
    '#description' => t('Allow experimental trackbacks sharing using Pingback 1.0 extended protocol (using XMLRPC pingback.extended.getPingbacks)'),
    '#default_value' => variable_get('trackback_receive_xmlrpc_getpingbacks', FALSE),
  );
  
  $form['trackback_moderation_settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('Moderation settings'),
    '#description' => t(''),
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );
  $form['trackback_moderation_settings']['trackback_moderation'] = array(
    '#type' => 'radios',
    '#title' => t('Use moderation queue?'),
    '#description' => t('If moderation queue is disabled, ALL trackbacks will be published. We suggest, if you disable, to activate a spam filtering module (more options for SPAM will be available)'),
    '#default_value' => variable_get('trackback_moderation', TRUE),
    '#options' => array(
      TRUE => 'Enable',
      FALSE => 'Disable'),
  );
  // If any module is enabled for trackback SPAM filtering
  if (trackback_spam_filtering()) {
    $form['trackback_moderation_settings']['trackback_spam_moderation'] = array(
      '#type' => 'radios',
      '#title' => t('Spam Moderation'),
      '#description' => t('If a Trackback is marked as SPAM'),
      '#default_value' => variable_get('trackback_spam_moderation', 'delete'),
      '#options' => array(
        'delete' => t ("Auto delete"),
        'approval' => t("Put into SPAM queue"),
      ),
    );
  }
  elseif(variable_get('trackback_spam_moderation', 'delete') == 'approval') {
    // Something is wrong here: no antispam and "mode to approval"?
    // Reset to "delete"
    variable_set('trackback_spam_moderation', 'delete');
  }

  // Sending settings
  $form['trackback_sending_settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('Sending Settings'),
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );
  
  $form['trackback_sending_settings']['trackback_sending'] = array(
    '#type' => 'radios',
    '#title' => t('Trackback senting setting'),
    '#description' => t(''),
    '#default_value' => variable_get('trackback_sending', 'oncron'),
    '#options' => array(
      'onupdate' => t('Send on node insert/update'),
      'oncron' => t('Send on cron run'),
    ),
  );
  
  // @TODO: Implement Sending settings (look at pingback module)
  
  // Auto-Detection settings
  // @TODO: review next part...
  $form['trackback_auto_detection_settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('Auto-detection'),    
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
    '#description' => t('Auto-detection for pings to other sites.'),
  );
  $form['trackback_auto_detection_settings']['trackback_auto_detection'] = array(
    '#type' => 'radios',
    '#title' => t('Auto-Detection settings'),
    '#description' => t(''),
    '#default_value' => variable_get('trackback_auto_detection', 'auto'),
    '#options' => array(
      'auto' => t('Auto-detect links from node body text (links and urls).'),
      'auto-links' => t('Auto-detect links from node body text (only links).'),
      'user' => t('Let user list URLs to ping'),
    ),
  );
  
  
  // @TODO: review next part...
  // Setting for Trackback display (trackbacks and Trackback-URL)
  $form['trackback_display'] = array(
    '#type' => 'fieldset',
    '#title' => t('Trackback display'),
    '#description' => t('Configure how and where received Trackbacks will be displayed.'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );

  // Location to display Trackbacks
  $form['trackback_display']['trackback_display_node'] = array(
    '#type' => 'checkbox',
    '#title' => t('Display trackbacks below node'),
    '#default_value' => variable_get('trackback_display_node', FALSE),
  );

  // Setting to place Trackback URL
  $form['trackback_display']['trackback_display_url'] = array(
    '#type' => 'radios',
    '#title' => t('Trackback URL display position'),
    '#description' => t('Setting for displayed trackback URL in node view. RDF meta-tags will always be added in HTML code. This setting is valid only for "POST/GET receiving" mode'),    
    '#default_value' => variable_get('trackback_display_url', 'innode'),
    '#options' => array(
      'none' => t('Hide from post'),
      'innode' => t('Below post'),
      'links' => t('Display in links'),
      'links-teaser' => t ('Display in links (only for teaser view)'),
      'links-full' =>   t ('Display in links (only for full-node view)')
    ),
  );
  
  // Setting to hide/show trackback counts in links
  $form['trackback_display']['trackback_display_link_count'] = array(
    '#type' => 'checkbox',
    '#title' => t('Display Trackback count in links'),
    '#default_value' => variable_get('trackback_display_link_count', FALSE),
  );  

  // Mass edit for node-type setting.
  $form['trackback_nodetypes'] = array(
    '#type' => 'fieldset',
    '#title' => t('Node types'),
    '#description' => t('Enable or disable Trackbacks for each node-type.'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );
  
  $form['trackback_nodetypes']['trackback_nodetypes'] = array(
    '#type' => 'checkboxes',      
    '#default_value' => variable_get('trackback_nodetypes', array()),
    '#title' => $type->name,
    '#options' => node_get_types('names'),
  );
  return system_settings_form($form);
}

/**
 * Check if a module for Trackback filtering is enabled
 */
function trackback_spam_filtering() {
  $null = NULL;
  $result = trackback_invoke_trackback('spam-filter', $null);
  return !empty($result);
}

/**
 *
 */
function trackback_translate_status($status = NULL, $reset = FALSE) {
  static $statuses = array();
  
  if (empty($statuses) || $reset) {
    $statuses[TRACKBACK_STATUS_PUBLISHED]   = t('Published');    
    $statuses[TRACKBACK_STATUS_UNPUBLISHED] = t('Unpublished');
    $statuses[TRACKBACK_STATUS_SPAM_UNSURE] = t('Spam Unsure');
    $statuses[TRACKBACK_STATUS_SPAM]        = t('Spam');
  }
  if ($status == NULL) {
    return $statuses;
  }
  else {
    return $statuses[$status];
  }
}

/**
 * Implementation of hook_help()
 */
function _trackback_help($path, $arg) {
  switch ($path) {
    //@TODO: review...
    case 'admin/help#trackback':
      $output = '<p>'. t('The trackback module allows users to give a blog post a contextual link to another. A context is made because the trackbacking poster is, in theory, writing about something mentioned on another blogger\'s trackbacked post.  Using the trackback URL accompanying each post, another website can send a ping to your website. Once received, the trackback will show up on the trackback page accompanying the post.  It also includes auto-discovery, spam moderation queues, and the ability to manually ping another site.') .'</p>';
      $output .= '<p>'. t('If trackback autodisovery is enabled on your website, someone need only visit your post via a link from another website post to have trackback <em>discover</em> the linking site and create the trackback. Trackback auto-discovery also works internally within a website, automatically creating connections between pages which link to each other. To manually send a ping to another site, edit your post and use the trackback URL field at the bottom of the edit page to submit the trackback URL for the post on the other site. Once you enter submit, your website will ping the other site for you. With trackback autodiscovery enabled, your site will attempt to do this automatically without your intervention.') .'</p>';
      $output .= '<p>'. t('To enable the moderation queue, go to the administer trackbacks page and select the configure tab. To view, approve, or delete trackbacks awaiting moderation, go to the administer trackbacks page and select the approval queue.  To administer the trackback settings for a particular content type go to that content types administration page.') .'</p>';
      $output .= '<p>'. t('For more information please read the configuration and customization handbook <a href="@trackback">Trackback page</a>.', array('@trackback' => 'http://www.drupal.org/handbook/modules/trackback')) .'</p>';
      return $output;
  }
}